<?php
/**
 * User Tag
 *
 * @package     AutomatorWP\Integrations\WP_Fusion\Filters\User_Tag
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WP_Fusion_User_Tag_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'wp_fusion';
    public $filter = 'wp_fusion_user_tag';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User tag', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>tag</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Condition. %2$s: Role. */
            'edit_label'        => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{tag}'  ),
            /* translators: %1$s: Condition. %2$s: Role. */
            'log_label'         => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{tag}' ),
            'options'           => array(
                'condition' => array(
                    'from' => 'condition',
                    'fields' => array(
                        'condition' => array(
                            'name' => __( 'Condition:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'equal'             => __( 'is equal to', 'automatorwp-pro' ),
                                'not_equal'         => __( 'is not equal to', 'automatorwp-pro' ),
                            ),
                            'default' => 'equal'
                        )
                    )
                ),
                'tag' => array(
                    'from' => 'tag',
                    'default' => __( 'Select a tag', 'automatorwp-pro' ),
                    'fields' => array(
                        'tag' => array(
                            'name' => __( 'Tag:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options_cb' => array( $this, 'options_cb_tags' ),
                            'default' => ''
                        )
                    )
                )
            ),
        ) );

    }

    /**
     * Options callback for tags options
     *
     * @since 1.0.0
     *
     * @param stdClass $field
     *
     * @return array
     */
    public function options_cb_tags( $field ) {

        $options = array();

        $tags = wp_fusion()->settings->get( 'available_tags' );

        if( is_array( $tags ) ) {
            $options = $tags;
        }

        return $options;

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        // Shorthand
        $condition = $filter_options['condition'];
        $tag = $filter_options['tag'];
        $tags = wp_fusion()->settings->get( 'available_tags' );

        // Bail if empty tag to assign
        if( empty( $tag ) ) {
            $this->result = __( 'Filter not passed. Tag option has not been configured.', 'automatorwp-pro' );
            return false;
        }

        // Get the tag ID
        $tag_id = wp_fusion()->user->get_tag_id( $tag );

        // Bail if tag doesn't exists
        if( $tag_id === false ) {
            $this->result = __( 'Filter not passed. Tag has not been registered in WP Fusion.', 'automatorwp-pro' );
            return false;
        }

        // Get contact ID
        $contact_id = wp_fusion()->user->get_contact_id( $user_id, true );

        // If user not registered then register him
        if ( $contact_id === false) {
            wp_fusion()->user->user_register( $user_id );
        }

        $user_tags = wp_fusion()->user->get_tags( $user_id );

        switch( $condition ) {
            case 'equal':
                // Don't deserve if user does not have this tag
                if( ! in_array( $tag_id, $user_tags, true ) ) {
                    $this->result = sprintf( __( 'Filter not passed. User does not have the tag "%1$s" assigned.', 'automatorwp-pro' ),
                        $tags[$tag]
                    );
                    return false;
                } else {
                    $this->result = sprintf( __( 'Filter passed. User has the tag "%1$s" assigned.', 'automatorwp-pro' ),
                        $tags[$tag]
                    );
                }
                break;
            case 'not_equal':
                // Don't deserve if user has this tag
                if( in_array( $tag_id, $user_tags, true ) ) {
                    $this->result = sprintf( __( 'Filter not passed. User has the tag "%1$s" assigned.', 'automatorwp-pro' ),
                        $tags[$tag]
                    );
                    return false;
                } else {
                    $this->result = sprintf( __( 'Filter passed. User does not have the tag "%1$s" assigned.', 'automatorwp-pro' ),
                        $tags[$tag]
                    );
                }
                break;
        }

        return $deserves_filter;

    }

}

new AutomatorWP_WP_Fusion_User_Tag_Filter();